#include "mex.h"
#include "matrix.h"
#include "math.h"
#include "stdlib.h"
 
/*
 *   Copyright (C) 2009 Cesare Magri
 *   Version: 1.0.2
 */

/*
 *-------
 * LICENSE
 * -------
 * This software is distributed free under the condition that:
 *
 * 1. it shall not be incorporated in software that is subsequently sold;
 *
 * 2. the authorship of the software shall be acknowledged and the following
 *    article shall be properly cited in any publication that uses results
 *    generated by the software:
 *
 *      Magri C, Whittingstall K, Singh V, Logothetis NK, Panzeri S: A
 *      toolbox for the fast information analysis of multiple-site LFP, EEG
 *      and spike train recordings. BMC Neuroscience 2009 10(1):81;
 *
 * 3.  this notice shall remain in place in each source file.
 *
 */

/* ----------
 * DISCLAIMER
 * ----------
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

void mexFunction(int nlhs, mxArray *plhs[],
                 int nrhs, const mxArray *prhs[])
{
    /* Input:                                                            */
    double *R;
    double *nt;
    double Npart;
    double partIndx;
    
    /* Dimensions of R:                                                  */
    mwSize Nc;
    mwSize maxNt;
    mwSize Ns;
    
    /* Indexes:                                                          */
    mwIndex c, t, s;
    
    mwIndex tmin;
    mwIndex tmax;

    mwSize maxNtOut;
    mwSize *outDims;
    
    /* Output:                                                           */
    double *Rout;
    double *totNtOut;
    
    /* Can be either an input or output (computed if not provided):      */
    double *ntOut;

    
    /* ----------------------------------------------------------------- */

    
    /* Read input:                                                       */
    R        = mxGetPr(prhs[0]);
    nt       = mxGetPr(prhs[1]);
    Npart    = *mxGetPr(prhs[2]);
    partIndx = *mxGetPr(prhs[3]);
    if(nrhs==5)
        ntOut = mxGetPr(prhs[4]);
    
    /* Get dimensions of R:                                              */
    Nc    = mxGetDimensions(prhs[0])[0];
    maxNt = mxGetDimensions(prhs[0])[1];
    /* If the users inputs a response matrix with a single stimulus then   */
    /* the number of dimensions for the response matrix will be 2 and      */
    /* asking for the number of elements in the third dimension will leak  */
    /* into a meaningless value. Thus first we need to check that there is */
    /* indeed a third dimension. We do this using MXGETNUMBEROFDIMENSIONS  */
    if(mxGetNumberOfDimensions(prhs[0])>2)
        Ns = mxGetDimensions(prhs[0])[2];
    else
        Ns = 1;
    
    /* Set dimensions of output:                                         */
    maxNtOut = maxNt / Npart; /* floored int ratio                       */
    
    outDims = mxCalloc(3, sizeof(mwSize));
    outDims[0] = Nc;
    outDims[1] = maxNtOut;
    outDims[2] = Ns;

    /* Allocate output                                                   */
    plhs[0] = mxCreateNumericArray(3, outDims, mxDOUBLE_CLASS, 0);
    Rout = mxGetPr(plhs[0]);
    
    /*
     * If ntOut is not provided as an input we need to build it together
     * with the total number of output trials:
     */
    if(nrhs<5) {
        plhs[1] = mxCreateDoubleMatrix(Ns, 1, mxREAL);
        ntOut = mxGetPr(plhs[1]);
        
        plhs[2] = mxCreateDoubleScalar(0);
        totNtOut = mxGetPr(plhs[2]);
    }

    /* Copy data into partition:                                         */
    for(s=0; s<Ns; s++)
    {
        /* if ntOut is not provided as an input we compute it:           */
        if(nrhs<5)
        {
            ntOut[s] = floor(nt[s] / Npart);
            *totNtOut += ntOut[s];
        }

        tmax = ntOut[s] * partIndx;
        tmin = tmax - ntOut[s];

        for(t=tmin; t<tmax; t++)
        {    
            for(c=0; c<Nc; c++)
            {
                Rout[c + (t-tmin)*Nc + s*Nc*maxNtOut] = R[c + t*Nc + s*Nc*maxNt];
            }
        }
    }
    
    mxFree(outDims);
}